
class GeminiService {
    constructor() {
        // Base URL
        this.baseUrl = "https://generativelanguage.googleapis.com/v1beta/models/";

        // Priority List of Models to Try
        this.modelCandidates = [
            "gemini-2.5-flash",
            "gemini-2.0-flash",
            "gemini-2.0-flash-001",
            "gemini-2.0-flash-lite-preview-02-05", // New lite model from user list
            "gemini-1.5-flash",
            "gemini-1.5-flash-001",
            "gemini-1.5-flash-8b",
            "gemini-2.0-flash-exp",
            "gemini-1.5-pro",
            "gemini-1.5-pro-001",
            "gemini-1.0-pro",
            "gemini-pro"
        ];

        this.activeModel = "gemini-2.5-flash"; // Default assumption updated
    }

    // --- 1. CORE: Connection Manager ---

    /**
     * Test a Key and Auto-Select the best Working Model
     * @param {string} key 
     * @param {function} callback ({valid: boolean, model: string, error: string})
     */
    validateConnection(key, callback) {
        if (!key) {
            callback({ valid: false, error: "Key is empty" });
            return;
        }

        // Recursive function to try models one by one
        const tryModel = (index) => {
            if (index >= this.modelCandidates.length) {
                // All specific models failed. Try listModels as last resort diagnostics.
                this.listModels(key, (listRes) => {
                    if (listRes.success) {
                        callback({ valid: false, error: "Key is valid but no generateContent models found. Available: " + listRes.models.join(", ") });
                    } else {
                        callback({ valid: false, error: "All models failed. Last Error: " + (lastError || "Unknown") + ". ListModels failed: " + listRes.error });
                    }
                });
                return;
            }

            const model = this.modelCandidates[index];
            const url = `${this.baseUrl}${model}:generateContent`;

            console.log(`Testing Model: ${model}...`);

            this.testRequest(key, url, (res) => {
                if (res.success) {
                    console.log(`✅ Success! Active Model: ${model}`);
                    this.activeModel = model;
                    this.saveConfig(key, model);
                    callback({ valid: true, model: model });
                } else {
                    lastError = `${model}: ${res.error}`;
                    console.warn(`❌ Model ${model} failed:`, res.error);
                    // Try next
                    tryModel(index + 1);
                }
            });
        };

        let lastError = "";
        // Start chain
        tryModel(0);
    }

    listModels(key, cb) {
        fetch(`https://generativelanguage.googleapis.com/v1beta/models?key=${key}`)
            .then(async res => {
                if (!res.ok) {
                    const txt = await res.text();
                    return cb({ success: false, error: `${res.status} ${txt}` });
                }
                const data = await res.json();
                const names = data.models ? data.models.map(m => m.name) : [];
                cb({ success: true, models: names });
            })
            .catch(e => cb({ success: false, error: e.message }));
    }

    testRequest(key, url, cb) {
        const testPayload = { contents: [{ parts: [{ text: "Hi" }] }] };
        fetch(`${url}?key=${key}`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(testPayload)
        })
            .then(async res => {
                if (res.ok) return cb({ success: true });
                // If 404, it means model not found -> definitely try next.
                // If 403, might be key issue, but could also be model access.
                const txt = await res.text();
                cb({ success: false, error: `${res.status} ${txt}` });
            })
            .catch(e => cb({ success: false, error: e.message }));
    }

    saveConfig(key, modelName) {
        const config = {
            gemini_api_key: key,
            gemini_model_name: modelName
        };
        chrome.storage.local.set(config);
    }

    // --- 2. PUBLIC API ---

    getKey(cb) {
        chrome.storage.local.get(['gemini_api_key', 'gemini_model_name'], (res) => {
            if (res.gemini_model_name) {
                this.activeModel = res.gemini_model_name;
            }
            cb(res.gemini_api_key);
        });
    }

    /**
     * Main Generation Function (Optimized)
     * Uses system_instruction and correct schemas
     */
    generateContent(options, callback) {
        const { prompt, systemInstruction, forceJson } = options;

        this.getKey((key) => {
            if (!key) return callback({ error: "API Key missing. Please check Settings." });

            const payload = {
                contents: [{ parts: [{ text: prompt }] }],
                safetySettings: [
                    { category: "HARM_CATEGORY_HARASSMENT", threshold: "BLOCK_ONLY_HIGH" },
                    { category: "HARM_CATEGORY_HATE_SPEECH", threshold: "BLOCK_ONLY_HIGH" },
                    { category: "HARM_CATEGORY_SEXUALLY_EXPLICIT", threshold: "BLOCK_ONLY_HIGH" },
                    { category: "HARM_CATEGORY_DANGEROUS_CONTENT", threshold: "BLOCK_ONLY_HIGH" }
                ]
            };

            // 1. System Instruction (New API Feature)
            if (systemInstruction) {
                payload.system_instruction = {
                    parts: [{ text: systemInstruction }]
                };
            }

            // 2. JSON Mode (New API Feature)
            if (forceJson) {
                payload.generationConfig = { responseMimeType: "application/json" };
            }

            const url = `${this.baseUrl}${this.activeModel}:generateContent`;

            fetch(`${url}?key=${key}`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(payload)
            })
                .then(res => res.json())
                .then(data => {
                    if (data.error) return callback({ error: data.error.message });

                    const text = data.candidates?.[0]?.content?.parts?.[0]?.text;
                    if (!text) return callback({ error: "No content generated." });

                    // JSON Handling
                    if (forceJson) {
                        try {
                            // If API returned pure JSON (as requested), parse directly
                            const json = JSON.parse(text);
                            callback({ result: json });
                        } catch (e) {
                            // Fallback cleanup if model was lazy or strict JSON mode failed
                            try {
                                const clean = text.replace(/```json/g, '').replace(/```/g, '').trim();
                                const json = JSON.parse(clean);
                                callback({ result: json });
                            } catch (e2) {
                                console.error("JSON Parse Fail:", text);
                                callback({ error: "AI output was not valid JSON." });
                            }
                        }
                    } else {
                        callback({ result: text });
                    }
                })
                .catch(err => callback({ error: "Network Error: " + err.message }));
        });
    }

    // Helper Wrappers for App
    generateSemanticKeywords(context, callback) {
        const systemInstruction = `Act as an SEO Expert. Return strictly a JSON Array of 30 to 50 unique LSI (Latent Semantic Indexing) keywords for the given title and headings. ensure they are semantically related but distinct. Output format: ["keyword1", "keyword2", ...]`;
        const prompt = `Title: "${context.title}"\nHeadings: [${context.headings.map(h => h.text).join(', ')}]`;

        this.generateContent({ prompt, systemInstruction, forceJson: true }, callback);
    }

    generateFullPageAnalysis(context, callback) {
        const systemInstruction = `Analyze the article context deeply. 
        1. Extract 30 to 50 unique NLP (Natural Language Processing) keywords/entities.
        2. Analyze E.E.A.T (Experience, Expertise, Authoritativeness, Trustworthiness) and provide a score (0-10) and brief justification.
        3. Determine Search Intent, Sentiment, and a Summary.
        
        Return strictly JSON Code: 
        { 
            "nlp_keywords": ["kw1", "kw2", ...], 
            "lsi_keywords": [], 
            "search_intent": "", 
            "sentiment": "", 
            "summary": "",
            "eeat": { "score": 0, "analysis": "" }
        }`;
        const prompt = `Title: "${context.title}"`;

        this.generateContent({ prompt, systemInstruction, forceJson: true }, callback);
    }
}
