/**
 * Auth Service
 * Handles interaction with Firebase Auth & Firestore
 */

const AuthService = {
    // Current User State
    user: null,
    userData: null, // Stores Firestore Profile (name, plan)
    isPro: false,
    // Initialize & Listen to Auth State
    init: function (callback) {
        if (typeof firebase === 'undefined') {
            console.error("Firebase not loaded");
            if (callback) callback(null);
            return;
        }

        this.db = firebase.firestore();

        firebase.auth().onAuthStateChanged(async (user) => {
            if (user) {
                this.user = user;
                // Fetch Pro Status
                this.isPro = await this.checkProStatus(user);
                console.log("Auth Init: User logged in, Pro:", this.isPro);
            } else {
                this.user = null;
                this.userData = null;
                this.isPro = false;
                console.log("Auth Init: User logged out");
            }
            if (callback) callback(this.user);
        });
    },

    // Check Pro Status from Firestore
    checkProStatus: async function (user) {
        if (!this.db) return false;

        try {
            const docRef = this.db.collection("users").doc(user.uid);
            const doc = await docRef.get();

            if (doc.exists) {
                const data = doc.data();
                this.userData = data; // Cache profile

                // AUTO-EXPIRATION CHECK
                if (data.isPro && data.expiresAt) {
                    // Handle Firestore Timestamp or Date String or number
                    let expiry;
                    if (data.expiresAt.toDate) {
                        expiry = data.expiresAt.toDate(); // Firestore Timestamp
                    } else if (data.expiresAt instanceof Date) {
                        expiry = data.expiresAt;
                    } else {
                        expiry = new Date(data.expiresAt); // String or Int
                    }

                    if (expiry < new Date()) {
                        console.log("Pro Plan Expired on:", expiry);
                        return false;
                    }
                }

                return data.isPro === true;
            } else {
                // First time setup
                await this.createUserDoc(user);
                return false;
            }
        } catch (e) {
            console.error("Error fetching user data:", e);
            return false;
        }
    },

    // Create User Document
    createUserDoc: async function (user, fname = '', lname = '', isPro = false) {
        if (!this.db) {
            console.error("Firestore DB missing during create");
            return;
        }
        try {
            console.log("Creating user doc for:", user.uid);
            await this.db.collection("users").doc(user.uid).set({
                email: user.email,
                firstName: fname,
                lastName: lname,
                isPro: isPro,
                createdAt: firebase.firestore.FieldValue.serverTimestamp()
            }, { merge: true });
            console.log("User doc created!");
        } catch (e) {
            console.error("Error creating user doc:", e);
        }
    },

    // Sign Up with Email/Pass
    register: async function (email, password, fname, lname, isPro) {
        try {
            const userCredential = await firebase.auth().createUserWithEmailAndPassword(email, password);
            // Create DB Entry immediately with details
            await this.createUserDoc(userCredential.user, fname, lname, isPro);

            // Cache locally immediately so UI works without reload
            this.userData = {
                firstName: fname,
                lastName: lname,
                isPro: isPro,
                email: email
            };

            return { success: true, user: userCredential.user };
        } catch (error) {
            return { success: false, error: error.message };
        }
    },

    // Login
    login: async function (email, password) {
        try {
            const userCredential = await firebase.auth().signInWithEmailAndPassword(email, password);
            return { success: true, user: userCredential.user };
        } catch (error) {
            return { success: false, error: error.message };
        }
    },

    // Logout
    logout: async function () {
        try {
            await firebase.auth().signOut();
            return { success: true };
        } catch (error) {
            return { success: false, error: error.message };
        }
    },

    // Password Reset
    resetPassword: async function (email) {
        try {
            await firebase.auth().sendPasswordResetEmail(email);
            return { success: true };
        } catch (error) {
            return { success: false, error: error.message };
        }
    }
};

// Expose to window
window.AuthService = AuthService;
