// SEO Pro - Advanced Logic

// State
let pageData = null;
let nlpKeywords = [];
let lsiKeywords = [];
let currentTab = 'tab-overview';
// Global variable to store AI results
let aiGeneratedResults = { keywords: [], intent: '', sentiment: '' };
// Service
const geminiService = new GeminiService(); // Instantiate globally

// ... (Rest of Lines 7-391 stay same, but we need to target specific chunks)

// Elements
const screens = {
    preview: document.getElementById('screen-preview'),
    dashboard: document.getElementById('screen-dashboard'),
    output: document.getElementById('screen-output')
};

const displays = {
    preview: document.getElementById('articlePreview'),
    error: document.getElementById('extractionError'),
    status: document.getElementById('statusBadge'),

    // Metrics
    words: document.getElementById('metricWords'),
    readability: document.getElementById('metricReadability'),
    time: document.getElementById('metricTime'),
    sentiment: document.getElementById('metricSentiment'),

    // Lists
    headings: document.getElementById('headingsList'),
    density: document.getElementById('densityList'),
    output: document.getElementById('finalOutput')
};

const inputs = {
    topic: document.getElementById('inputTopic'),
    primary: document.getElementById('inputPrimaryKw'),
    secondary: document.getElementById('inputSecondaryKw'),
    nlp: document.getElementById('inputNLPKw'),
    template: document.getElementById('promptTemplate')
};

// --- Auth / Freemium Logic ---

// --- AUTHENTICATION & LOCKS ---

// --- AUTHENTICATION & LOCKS ---

function setupAuth() {
    const btn = document.getElementById('btnLoginAction'); // Header Button
    const overlay = document.getElementById('loginOverlay');
    const btnCancel = document.getElementById('btnCancelLogin');
    const btnWebLogin = document.getElementById('btnWebLogin');

    // 0. CHECK SYNCED USER FIRST & LISTEN FOR UPDATES
    function checkUser() {
        chrome.storage.local.get(['user'], (result) => {
            if (result.user) {
                console.log("Found synced user:", result.user.email);
                AuthService.user = result.user;
                AuthService.isPro = (result.user.plan === 'pro');
                updateUILocks();
                setupHeaderUser();
                if (overlay) overlay.style.display = 'none'; // Auto close if open
            } else {
                // No user found
                AuthService.user = null;
                AuthService.isPro = false;
                updateUILocks();
                setupHeaderUser();
            }
        });
    }

    // Initial Check
    checkUser();

    // Listen for storage changes (Real-time sync from background)
    chrome.storage.onChanged.addListener((changes, namespace) => {
        if (namespace === 'local' && changes.user) {
            console.log("Storage changed (User update). Refreshing UI...");
            checkUser();
        }
    });

    if (!overlay) {
        console.warn("Login overlay missing");
        return;
    }

    // 1. Open Overlay
    if (btn) {
        btn.onclick = () => {
            overlay.style.display = 'flex';
        };
    }

    // 2. Close Overlay
    if (btnCancel) {
        btnCancel.onclick = () => {
            overlay.style.display = 'none';
        };
    }

    // 3. Web Login Action
    if (btnWebLogin) {
        btnWebLogin.onclick = () => {
            chrome.tabs.create({ url: 'https://app.survivalenginepro.com/auth.html' });
            // Overlay stays open or closes, user choice. Let's close it so they see the new tab.
            overlay.style.display = 'none';
        };
    }
}

function setupHeaderUser() {
    const user = (typeof AuthService !== 'undefined') ? AuthService.user : null;
    let isPro = (typeof AuthService !== 'undefined') ? AuthService.isPro : false;


    const controls = document.getElementById('loggedInControls');
    const btnLogin = document.getElementById('btnLoginAction');
    const btnUpgrade = document.getElementById('btnUpgradePro');
    const nameSpan = document.getElementById('headerUserName');
    const btnSignOut = document.getElementById('btnSignOut');

    // 1. HEADER UPGRADE BUTTON
    if (btnUpgrade) {
        if (!isPro) {
            btnUpgrade.style.display = 'block';

            // Re-assign listener to ensure it works
            btnUpgrade.replaceWith(btnUpgrade.cloneNode(true));
            const activeUpgradeBtn = document.getElementById('btnUpgradePro');

            activeUpgradeBtn.addEventListener('click', () => {
                const ov = document.getElementById('loginOverlay');
                const tabSign = document.getElementById('tabSignup');
                if (ov) {
                    ov.style.display = 'flex';
                    // Check if tabSignup exists in new legacy mode, otherwise just set mode
                    if (tabSign) {
                        tabSign.click();
                    } else {
                        // Fallback if tab element hidden/removed
                        // Just ensure plan selection is shown if possible
                        // But since we reverted, tabSignup should be there.
                    }
                }
            });
        } else {
            btnUpgrade.style.display = 'none';
        }
    }

    // 2. AUTH CONTROLS
    if (!controls) return;

    if (user) {
        // Show Controls, Hide Login Button
        controls.style.display = 'flex';
        if (btnLogin) btnLogin.style.display = 'none';

        // Set Name
        // Logic: Try userData.firstName, else split displayName, else "User"
        let fname = "User";
        if (AuthService.userData && AuthService.userData.firstName) {
            fname = AuthService.userData.firstName;
        } else if (user.displayName) {
            fname = user.displayName.split(' ')[0];
        }

        if (nameSpan) nameSpan.textContent = fname;

        // Sign Out Listener
        if (btnSignOut) {
            const newSignOut = btnSignOut.cloneNode(true);
            btnSignOut.parentNode.replaceChild(newSignOut, btnSignOut);

            newSignOut.addEventListener('click', () => {
                AuthService.logout().then(() => {
                    window.location.reload();
                });
            });
        }

    } else {
        // Hide Controls, Show Login Button
        controls.style.display = 'none';
        if (btnLogin) {
            btnLogin.style.display = 'block';
            btnLogin.textContent = "Login 🔐";
        }
    }
}

function updateUILocks() {
    const btn = document.getElementById('btnLogin');
    const lockOverlay = document.getElementById('proLockOverlay');
    const promptBtn = document.getElementById('btnGoToPrompt');

    const user = (typeof AuthService !== 'undefined') ? AuthService.user : null;
    const isPro = (typeof AuthService !== 'undefined') ? AuthService.isPro : false;

    // 1. Header Button Text
    if (user) {
        const email = user.email ? user.email.split('@')[0] : 'User';
        btn.textContent = email + (isPro ? " (Pro)" : " (Free)");
        btn.style.background = isPro ? "#10b981" : "#475569"; // Green if Pro, Slate if Free
        btn.style.cursor = "pointer";
        btn.title = "Click to Logout";
    } else {
        btn.textContent = "Login";
        btn.style.background = "#1e293b"; // Dark
        btn.style.cursor = "pointer";
        btn.title = "Login to unlock features";
    }

    // 2. Feature Gating
    if (isPro) {
        // PRO MODE
        if (lockOverlay) lockOverlay.style.display = 'none';

        if (promptBtn) {
            promptBtn.textContent = "Open Prompt Generator";
            promptBtn.disabled = false;
            promptBtn.style.opacity = "1";
            promptBtn.style.cursor = "pointer";
        }
    } else {
        // FREE MODE (Logged In OR Logged Out)
        if (lockOverlay) {
            lockOverlay.style.display = 'flex';
            const title = lockOverlay.querySelector('h3');
            const desc = lockOverlay.querySelector('p');
            if (title && desc) {
                if (user) {
                    title.textContent = "Pro Plan Required";
                    desc.textContent = "Your account is currently on the Free tier.";
                } else {
                    title.textContent = "Pro Feature";
                    desc.textContent = "Login or Upgrade to unlock NLP & LSI Keywords";
                }
            }
        }

        if (promptBtn) {
            promptBtn.textContent = "Open Content Studio"; // Allow free users to see this button
            promptBtn.disabled = false; // Allow free users to click
            promptBtn.style.opacity = "1";
            promptBtn.style.cursor = "pointer";
        }
    }
}

// --- Init ---
const EXT_VERSION = '3.5.0'; // Increment to force cache clear

document.addEventListener('DOMContentLoaded', () => {
    // 1. CACHE CLEAR CHECK
    chrome.storage.local.get(['ext_version'], (res) => {
        if (res.ext_version !== EXT_VERSION) {
            console.log("⚠️ New Version Detected! Clearing Cache...");
            chrome.storage.local.remove(['cachedAnalysis', 'seoData', 'gemini_model_type'], () => {
                chrome.storage.local.set({ 'ext_version': EXT_VERSION });
                console.log("✅ Cache Cleared. Starting Fresh.");
            });
        }
    });



    // 2. FAIL-SAFE INIT (Wrap components so one crash doesn't kill app)
    try { setupAuth(); } catch (e) { console.error("⚠️ Auth Init Failed:", e); }
    try { setupHandlers(); } catch (e) { console.error("⚠️ Handlers Init Failed:", e); }
    try { setupTabs(); } catch (e) { console.error("⚠️ Tabs Init Failed:", e); }

    // RESTORE STATE: Check if validity matches current tab
    chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
        const currentUrl = tabs[0]?.url;

        chrome.storage.local.get(['cachedAnalysis'], (result) => {
            if (result.cachedAnalysis && currentUrl) {
                // Normalize URL comparison (ignoring fragments if needed, but strict is fine for now)
                if (result.cachedAnalysis.url === currentUrl) {
                    console.log("Restoring cached analysis for current URL...");
                    processData(result.cachedAnalysis);
                } else {
                    console.log("Cache mismatch (Different URL). Pending fresh analysis.");
                    // Optionally clear cache or just do nothing
                }
            }
        });
    });
});

function setupHandlers() {
    document.getElementById('btnExtract').addEventListener('click', onExtract);

    // Navigation
    document.getElementById('btnGoToPrompt').addEventListener('click', saveAndOpenGenerator);
    const btnGoToAi = document.getElementById('btnGoToAiPrompt');
    if (btnGoToAi) btnGoToAi.addEventListener('click', secureOpenAiGenerator);

    document.getElementById('btnBackToPreview').addEventListener('click', () => switchScreen('preview'));
    document.getElementById('btnDomainAge').addEventListener('click', checkDomainAge);

    // Settings (v2.2)
    const btnSettings = document.getElementById('btnSettings');
    if (btnSettings) btnSettings.addEventListener('click', () => toggleSettings(true));

    // Close Actions (X)
    const btnCloseX = document.getElementById('btnCloseSettingsX');
    if (btnCloseX) btnCloseX.addEventListener('click', () => toggleSettings(false));

    // Test Button
    const btnTest = document.getElementById('btnTestConnection');
    if (btnTest) btnTest.addEventListener('click', handleTestConnection);

    // Save Button
    const btnSaveKey = document.getElementById('btnSaveKey');
    if (btnSaveKey) btnSaveKey.addEventListener('click', saveGeminiKey);

    // Clear Button
    const btnClear = document.getElementById('btnClearKey');
    if (btnClear) btnClear.addEventListener('click', clearGeminiKey);

    // AI Action (v2.2)
    const btnAi = document.getElementById('btnAiKeywords');
    if (btnAi) btnAi.addEventListener('click', runAiAnalysis);

    // AI Buttons (Both Regenerate & Activate)
    const btnAiBrain = document.getElementById('btnAiBrainActivate');
    if (btnAiBrain) btnAiBrain.addEventListener('click', runAiAnalysis);

    // Copy Buttons
    setupCopyBtn('btnCopyFocus', 'inputPrimaryKw'); // NEW
    setupCopyBtn('btnCopyNLP', 'inputNLPKw');
    setupCopyBtn('btnCopyLSI', 'inputSecondaryKw');

    const btnCopyHeadings = document.getElementById('btnCopyHeadings');
    if (btnCopyHeadings) {
        btnCopyHeadings.addEventListener('click', () => {
            const text = pageData.headings.map(h => `${h.tag}: ${h.text}`).join('\n');
            navigator.clipboard.writeText(text).then(() => {
                btnCopyHeadings.textContent = "Copied!";
                setTimeout(() => btnCopyHeadings.textContent = "Copy", 2000);
            });
        });
    }

    // Initial View
    switchScreen('preview');
}

function setupCopyBtn(btnId, targetId) {
    const btn = document.getElementById(btnId);
    if (!btn) return;

    btn.addEventListener('click', () => {
        const target = document.getElementById(targetId);
        if (target) {
            target.select();
            document.execCommand('copy'); // Fallback or use navigator.clipboard

            const original = btn.textContent;
            btn.textContent = "Copied!";
            setTimeout(() => btn.textContent = original, 2000);
        }
    });
}

function switchScreen(screenName) {
    console.log("Switching to screen:", screenName);

    // Hide all
    Object.values(screens).forEach(s => {
        if (s) {
            s.classList.remove('active');
            s.style.display = 'none';
        }
    });

    // Show target
    const target = screens[screenName];
    if (target) {
        target.classList.add('active');
        target.style.display = 'block';
    } else {
        console.error("Screen not found:", screenName);
    }
}

// --- CORRECTED SETTINGS LOGIC ---
function toggleSettings(show) {
    const overlay = document.getElementById('settingsOverlay');
    if (overlay) {
        overlay.style.display = show ? 'flex' : 'none';
        if (show) {
            // Pre-fill key
            if (typeof geminiService !== 'undefined') {
                geminiService.getKey(key => {
                    const input = document.getElementById('inputGeminiKey');
                    if (key && input) input.value = key;
                });
            } else {
                // Fallback read
                chrome.storage.local.get(['gemini_api_key'], (res) => {
                    const input = document.getElementById('inputGeminiKey');
                    if (res.gemini_api_key && input) input.value = res.gemini_api_key;
                });
            }
        }
    }
}

// Actions
document.getElementById('btnCopy').addEventListener('click', copyOutput);
document.getElementById('btnDownload').addEventListener('click', downloadReport);

// v1.3 Batch 2: Export CSV
const btnExport = document.getElementById('btnExportCSV');
if (btnExport) btnExport.addEventListener('click', exportCsv);

// Prompt Logic Binds
inputs.template.addEventListener('change', updateInternalPrompt);

// Upgrade Button
const btnUpgrade = document.getElementById('btnUpgrade');
if (btnUpgrade) {
    btnUpgrade.addEventListener('click', () => {
        if (typeof PaymentConfig !== 'undefined' && PaymentConfig.checkoutUrl) {
            chrome.tabs.create({ url: PaymentConfig.checkoutUrl });
        } else {
            alert("Payment link not configured yet.");
        }
    });
}

// END setupHandlers

function saveGeminiKey() {
    const input = document.getElementById('inputGeminiKey');
    const key = input ? input.value.trim() : "";
    const btn = document.getElementById('btnSaveKey');

    if (!key) {
        alert("Please enter a key.");
        return;
    }

    if (btn) {
        btn.textContent = "Verifying...";
        btn.disabled = true;
    }

    // Use Service to Validate & Auto-Select Model
    if (typeof geminiService !== 'undefined') {
        geminiService.validateConnection(key, (res) => {
            if (btn) btn.disabled = false;

            if (res.valid) {
                // Success UI
                if (btn) {
                    btn.textContent = "Saved! ✅";
                    btn.style.background = "#059669";

                    setTimeout(() => {
                        btn.textContent = "Save Settings";
                        btn.style.background = "#2563eb";
                        toggleSettings(false);
                    }, 1500);
                } else {
                    alert(`Success! Connected to ${res.model}`);
                    toggleSettings(false);
                }
            } else {
                // Failure UI
                if (btn) {
                    btn.textContent = "Failed ❌";
                    btn.style.background = "#ef4444";
                    setTimeout(() => {
                        btn.textContent = "Save Settings";
                        btn.style.background = "#2563eb";
                    }, 2000);
                }
                alert("Connection Failed: " + (res.error || "Unknown Error"));
            }
        });
    } else {
        // Fallback if service missing
        chrome.storage.local.set({ 'gemini_api_key': key }, () => {
            alert("Key Saved (Validation Skipped)");
            toggleSettings(false);
        });
    }
}

// Renamed to match event listener binding
function saveAndOpenGenerator() {
    if (typeof AuthService === 'undefined' || !AuthService.isPro) {
        alert("🔒 Pro Feature: Please upgrade or login to access the Advanced Prompt Generator.");
        return; // Gatekeeper
    }

    const topic = inputs.topic.value;
    const nlp = document.getElementById('inputNLPKw').value;
    const lsi = inputs.secondary.value;

    // Save data to storage
    chrome.storage.local.set({
        seoData: {
            title: topic,
            nlp: nlp,
            lsi: lsi,
            headings: pageData ? pageData.headings : [],
            intent: aiGeneratedResults.intent || '',
            sentiment: aiGeneratedResults.sentiment || ''
        }
    }, () => {
        // OPEN THE NEW AI PAGE INSTEAD of prompt_generator.html
        chrome.tabs.create({ url: 'ai_prompt_generator.html' });
    });
}

function setupTabs() {
    const nav = document.querySelector('.tabs-nav');
    if (!nav) return;

    nav.addEventListener('click', (e) => {
        const btn = e.target.closest('.tab-btn');
        if (!btn) return;

        // 1. Get Target
        const targetId = btn.getAttribute('data-tab');
        if (!targetId) return;

        console.log("Tab Clicked:", targetId);

        // 2. Deactivate All
        document.querySelectorAll('.tab-btn').forEach(b => b.classList.remove('active'));
        document.querySelectorAll('.tab-content').forEach(c => c.classList.remove('active'));

        // 3. Activate Clicked
        btn.classList.add('active');
        const targetContent = document.getElementById(targetId);
        if (targetContent) {
            targetContent.classList.add('active');
        } else {
            console.error("Tab Target Not Found:", targetId);
        }
    });
}

function switchScreen(name) {
    Object.values(screens).forEach(s => s.classList.remove('active'));
    screens[name].classList.add('active');
}

// --- Extraction & Analysis ---

async function onExtract() {
    displays.error.style.display = 'none';
    document.getElementById('btnExtract').textContent = "Analyzing...";

    try {
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        if (!tab) throw new Error("No active tab.");

        // Communicate with Content Script
        const isPro = (typeof AuthService !== 'undefined') ? AuthService.isPro : false;

        chrome.tabs.sendMessage(tab.id, { action: 'extract_content', isPro: isPro }, (response) => {
            if (chrome.runtime.lastError) {
                // Inject if needed
                chrome.scripting.executeScript({
                    target: { tabId: tab.id },
                    files: ['content.js']
                }, () => {
                    chrome.tabs.sendMessage(tab.id, { action: 'extract_content', isPro: isPro }, processData);
                });
            } else {
                processData(response);
            }
        });
    } catch (e) {
        showError(e.message);
    }
}

function processData(data) {
    try {
        console.log("Processing Data:", data);
        document.getElementById('btnExtract').textContent = "Analyze Article";

        if (!data || !data.valid) {
            showError(data?.error || "Extraction failed.");
            return;
        }

        pageData = data;
        inputs.topic.value = data.title;

        // PERSISTENCE: Save analysis to storage
        chrome.storage.local.set({ cachedAnalysis: data });

        // 0. Calculations
        const readabilityScore = calculateReadability(data.rawText);
        const sent = calculateSentiment(data.rawText);
        const readTime = Math.ceil(data.wordCount / 200);

        // 1. FAST RENDER: Show Local Data Immediately (Zero Wait)
        displayOverviewMetrics({
            words: data.wordCount,
            readability: readabilityScore,
            time: readTime + "m",
            sentiment: sent
        });

        document.getElementById('techTitleVal').innerText = data.title || "No Title Found";

        // Run Local Analysis (Regex) - Guaranteed result
        runLocalAnalysis(data, sent);

        // 2. BACKGROUND UPGRADE: Check for AI
        if (typeof geminiService !== 'undefined') {
            geminiService.getKey((key) => {
                if (key) {
                    // Upgrade to AI Analysis silently
                    console.log("Gemini Key found, upgrading analysis...");
                    runAIAnalysis(data, sent);
                } else {
                    console.log("No Gemini Key, staying on local analysis.");
                }
            });
        }
    } catch (e) {
        console.error("CRITICAL ERROR:", e);
        alert("System Error: " + e.message + "\n" + e.stack);
    }
}

// =========================================================
// [A] AI ANALYSIS (Upgrade)
// =========================================================
function runAIAnalysis(data, fallbackSentiment) {
    // 1. Loading Indicator (Subtle)
    const intentBadge = document.getElementById('intentBadge');
    intentBadge.textContent = "AI Optimizing... 🤖";
    intentBadge.style.backgroundColor = "#e0e7ff";
    intentBadge.style.color = "#4338ca";

    // --- CACHING STRATEGY (Protect Quota) ---
    // Create a simple fingerprint
    const cacheKey = "ai_cache_" + btoa(encodeURIComponent(data.title)).substring(0, 50);

    chrome.storage.local.get([cacheKey], (result) => {
        if (result[cacheKey]) {
            console.log("⚡ Cache Hit! Loading saved AI analysis...");
            applyAiResults(result[cacheKey], fallbackSentiment);
            return;
        }

        // Cache Miss -> Call API
        geminiService.generateFullPageAnalysis({
            title: data.title,
            content: data.rawText,
            headings: data.headings
        }, (response) => {
            if (response.error) {
                console.warn("AI Analysis Failed:", response.error);

                // Check for Quota/Rate Limit
                const errStr = String(response.error).toLowerCase();
                const isQuota = errStr.includes('quota') || errStr.includes('429');

                if (isQuota) {
                    intentBadge.textContent = "AI Busy (Quota) ⏳";
                    intentBadge.style.backgroundColor = "#fff7ed";
                    intentBadge.style.color = "#c2410c";
                    document.getElementById('inputNLPKw').value += " (Limit Reached - Try again later)";
                } else {
                    const localIntent = analyzeIntent(data.title, data.headings);
                    intentBadge.textContent = localIntent.type;
                    intentBadge.style.backgroundColor = localIntent.bg;
                    intentBadge.style.color = localIntent.fg;
                }
                return;
            }

            // Success -> Save to Cache
            const ai = response.result;
            // Save for future (limit storage by simple overwriting)
            chrome.storage.local.set({ [cacheKey]: ai });

            applyAiResults(ai, fallbackSentiment);
        });
    });
}

function applyAiResults(ai, fallbackSentiment) {
    console.log("Applying AI Results:", ai);
    const intentBadge = document.getElementById('intentBadge');

    // Keywords (Overwrite Local)
    const nlpText = ai.nlp_keywords ? ai.nlp_keywords.join(', ') : "";
    const lsiText = ai.lsi_keywords ? ai.lsi_keywords.join(', ') : "";

    document.getElementById('inputNLPKw').value = nlpText;
    document.getElementById('inputSecondaryKw').value = lsiText;

    // Save to Global for Prompt Gen
    aiGeneratedResults = {
        keywords: [...(ai.nlp_keywords || []), ...(ai.lsi_keywords || [])],
        intent: ai.search_intent,
        sentiment: ai.sentiment || fallbackSentiment.label
    };

    // Intent
    if (ai.search_intent) {
        intentBadge.textContent = "AI: " + ai.search_intent;
        const iLower = ai.search_intent.toLowerCase();
        if (iLower.includes('info')) { intentBadge.style.backgroundColor = '#d1fae5'; intentBadge.style.color = '#059669'; }
        else if (iLower.includes('trans') || iLower.includes('buy') || iLower.includes('comm')) { intentBadge.style.backgroundColor = '#fef3c7'; intentBadge.style.color = '#d97706'; }
        else { intentBadge.style.backgroundColor = '#e0e7ff'; intentBadge.style.color = '#4338ca'; }
    }

    // Sentiment
    if (ai.sentiment) {
        const sentEl = document.getElementById('metricSentiment');
        sentEl.textContent = ai.sentiment;
        if (ai.sentiment.toLowerCase().includes('positive')) sentEl.style.color = "#10b981";
        else if (ai.sentiment.toLowerCase().includes('negative')) sentEl.style.color = "#ef4444";
        else sentEl.style.color = "#64748b";
    }
}

// =========================================================
// [B] LOCAL ANALYSIS (Regex)
// =========================================================
function runLocalAnalysis(data, sent) {
    console.log("Running Local Analysis...");

    // Keyword Logic
    const titleTokens = cleanAndTokenize(data.title);
    const bodyTokens = cleanAndTokenize(data.rawText);

    let rawNLP = extractNLPKeywords(data.rawText, data.headings, titleTokens);
    let rawLSI = extractLSIKeywords(bodyTokens);

    nlpKeywords = removeRedundantPhrases(rawNLP);
    lsiKeywords = removeRedundantPhrases(rawLSI);

    // Filter Cross-Dedup
    const nlpSet = new Set(nlpKeywords.map(k => k.word));
    lsiKeywords = lsiKeywords.filter(k => !nlpSet.has(k.word));

    // Limit if needed (Free)
    const isPro = (typeof AuthService !== 'undefined' && AuthService.isPro);
    if (!isPro) {
        nlpKeywords = nlpKeywords.slice(0, 20);
        lsiKeywords = lsiKeywords.slice(0, 20);
    }

    // Display keywords
    const topNLP = nlpKeywords.slice(0, 40).map(k => k.word).join(', ');
    const topLSI = lsiKeywords.slice(0, 40).map(k => k.word).join(', ');

    document.getElementById('inputNLPKw').value = topNLP;
    document.getElementById('inputSecondaryKw').value = topLSI;

    // Intent
    const intent = analyzeIntent(data.title, data.headings);
    const intentBadge = document.getElementById('intentBadge');
    intentBadge.textContent = intent.type;
    intentBadge.style.backgroundColor = intent.bg;
    intentBadge.style.color = intent.fg;

    // Secondary
    updateDensityVisuals(data.rawText);
    updateSecondaryMetrics(data, nlpKeywords[0]?.word || "");
}

// Shared UI Updater for Density
function updateDensityVisuals(text) {
    const density = calculateKeywordDensity(text);
    renderDensityChart(density);
}

// Shared UI Updater for Tech/Survival (The rest of the dashboard)
function updateSecondaryMetrics(data, primaryKw) {
    const inputs = { primary: { value: primaryKw } }; // Mock inputs for legacy helper access if needed

    // B. HEADINGS (Structure)
    const headingsDiv = document.getElementById('headingsList');
    if (headingsDiv && data.headings) {
        if (data.headings.length === 0) {
            headingsDiv.innerHTML = '<div style="padding:10px; color:#94a3b8; font-size:12px; text-align:center;">No headings found (h1-h6)</div>';
        } else {
            headingsDiv.innerHTML = data.headings.map(h => {
                const indent = h.tag === 'H1' ? 0 : (h.tag === 'H2' ? 0 : (h.tag === 'H3' ? 10 : 20));
                // Tag Color
                let tagColor = '#64748b';
                if (h.tag === 'H1') tagColor = '#2563eb';
                if (h.tag === 'H2') tagColor = '#0ea5e9';
                if (h.tag === 'H3') tagColor = '#6366f1';

                return `
                <div style="margin-bottom:6px; margin-left:${indent}px; font-size:12px; display:flex; gap:6px; align-items:baseline;">
                    <span style="font-size:9px; font-weight:700; color:${tagColor}; background:#f1f5f9; padding:1px 4px; border-radius:3px; min-width:20px; text-align:center;">${h.tag}</span>
                    <span style="color:#334155; line-height:1.4;">${h.text}</span>
                </div>`;
            }).join('');
        }
    }

    // C. E-E-A-T
    const eeat = analyzeEEAT(data.eeat || {});
    document.getElementById('eeatScore').textContent = `${eeat.score}/100`;
    document.getElementById('eeatScore').style.color = eeat.color;
    document.getElementById('eeatSignals').innerHTML = eeat.signals.map(s => `<div>${s}</div>`).join('');

    // C. Schema
    const schemaDiv = document.getElementById('schemaList');
    if (data.schemas && data.schemas.length > 0) {
        const uniqueSchemas = [...new Set(data.schemas)];
        schemaDiv.innerHTML = uniqueSchemas.map(s =>
            `<span class="badge-tag" style="background:#f1f5f9; color:#475569; font-size:10px;">${s}</span>`
        ).join('');
    } else {
        schemaDiv.innerHTML = `<span style="font-size:10px; color:#ef4444;">No Structured Data Found</span>`;
    }

    // --- v2.0 SURVIVAL SUITE UI ---
    const human = analyzeHumanScore(data.rawText);
    document.getElementById('humanScoreVal').textContent = `${human.score}%`;
    document.getElementById('humanScoreBar').style.width = `${human.score}%`;
    document.getElementById('humanScoreBar').style.backgroundColor = human.score > 70 ? '#10b981' : (human.score > 40 ? '#f59e0b' : '#ef4444');
    document.getElementById('humanMsg').textContent = human.msg;

    const expSignals = analyzeExperience(data.rawText);
    document.getElementById('experienceSignals').innerHTML = expSignals.map(s => {
        const color = s.startsWith('✔') ? '#10b981' : '#cbd5e1';
        const txt = s.startsWith('✔') ? s : `<span style="opacity:0.6">${s}</span>`;
        return `<div style="color:#334155; font-size:11px;">${txt}</div>`;
    }).join('');

    // Risk Radar
    const freshness = analyzeFreshness(data.rawText);
    const seoLook = analyzeSeoLook(data.rawText, data.headings, primaryKw);
    const aio = analyzeAiOverview(data.rawText, data.headings);
    const scaled = analyzeScaledRisk(data.rawText);
    const volatility = analyzeVolatility(freshness, human.score);

    const riskList = document.getElementById('riskList');
    riskList.innerHTML = `
        <div style="display:flex; justify-content:space-between; font-size:11px;">
           <span>Freshness Risk</span>
           <span style="color:${freshness.score < 50 ? '#ef4444' : '#10b981'}">${freshness.status}</span>
        </div>
        <div style="display:flex; justify-content:space-between; font-size:11px;">
           <span>SEO-Look Risk</span>
           <span style="color:${seoLook.risk === 'High' ? '#ef4444' : (seoLook.risk === 'Medium' ? '#f59e0b' : '#10b981')}">${seoLook.risk}</span>
        </div>
         <div style="display:flex; justify-content:space-between; font-size:11px;">
           <span>Scaled Content Risk</span>
           <span style="color:${scaled.risk === 'High' ? '#ef4444' : (scaled.risk === 'Medium' ? '#f59e0b' : '#10b981')}">${scaled.risk}</span>
        </div>
        <div style="display:flex; justify-content:space-between; font-size:11px; margin-top:4px; border-top:1px dashed #e2e8f0; padding-top:4px;">
           <span>Volatility Immunity</span>
           <span style="font-weight:600; color:#6366f1;">${volatility}</span>
        </div>
        <div style="font-size:10px; color:#64748b; font-style:italic; margin-top:4px;">${seoLook.msg} ${scaled.msg}</div>
    `;

    document.getElementById('aioScore').textContent = `${aio.score}%`;
    document.getElementById('aioMsg').textContent = aio.msg;

    // Technical
    if (data.tech) {
        const tech = data.tech;
        const fmtDate = (d) => d ? new Date(d).toLocaleDateString() : "Not found";
        document.getElementById('techPubDate').textContent = fmtDate(tech.datePublished);
        document.getElementById('techModDate').textContent = fmtDate(tech.dateModified);

        // Indexing
        const canonicalEl = document.getElementById('techCanonical');
        if (tech.canonical === window.location.href) {
            canonicalEl.innerHTML = `<span style="color:#10b981">Self-referencing</span>`;
        } else if (tech.canonical) {
            canonicalEl.textContent = tech.canonical;
            canonicalEl.style.fontSize = "10px";
        } else {
            canonicalEl.innerHTML = `<span style="color:#ef4444">Missing</span>`;
        }

        document.getElementById('techRobots').textContent = tech.robots || "Index, Follow (Default)";
        document.getElementById('techHreflang').textContent = `${tech.hreflangs.length} tags found`;

        // Hreflangs
        const hrefBox = document.getElementById('techHreflangBox');
        if (tech.hreflangs.length > 0) {
            hrefBox.style.display = 'block';
            document.getElementById('techHreflangList').innerHTML = tech.hreflangs.map(l =>
                `<div style="font-size:11px; display:flex; justify-content:space-between;">
                    <span style="font-weight:600; color:#6366f1;">${l.lang}</span>
                    <span style="color:#64748b; max-width:180px; overflow:hidden; text-overflow:ellipsis; white-space:nowrap;">${l.href}</span>
                 </div>`
            ).join('');
        } else if (hrefBox) {
            hrefBox.style.display = 'none';
        }

        // Meta
        const titleLen = tech.metaTitleLen;
        document.getElementById('techTitleLen').innerHTML = `<span style="color:${titleLen < 60 ? '#10b981' : '#ef4444'}">${titleLen} / 60 chars</span>`;

        const descLen = tech.metaDescLen;
        document.getElementById('techDescLen').innerHTML = `<span style="color:${descLen >= 150 && descLen <= 165 ? '#10b981' : '#f59e0b'}">${descLen} / 160 chars</span>`;
        if (document.getElementById('techDescVal')) document.getElementById('techDescVal').textContent = tech.metaDescription || "(No Meta Description found)";

        // Images
        if (document.getElementById('techImgTotal')) document.getElementById('techImgTotal').textContent = tech.images.total;
        const imgAlt = document.getElementById('techImgAlt');
        if (tech.images.missingAlt > 0) {
            imgAlt.textContent = `${tech.images.missingAlt} Missing`;
            imgAlt.style.color = '#ef4444';
        } else {
            imgAlt.textContent = "0 (Perfect)";
            imgAlt.style.color = '#10b981';
        }

        // Social
        document.getElementById('techOgTitle').textContent = tech.ogTitle ? "✅ Found" : "❌ Missing";
        document.getElementById('techOgImage').textContent = tech.ogImage ? "✅ Found" : "❌ Missing";
        document.getElementById('techTwitter').textContent = tech.twitterCard ? `✅ ${tech.twitterCard}` : "❌ Missing";
    }

    // Snippet Opp
    const intent = analyzeIntent(data.title, data.headings);
    const intentType = intent ? intent.type : "Informational";
    const snippet = analyzeSnippetOpp(intentType, data.headings, data.title);
    document.getElementById('snippetOpp').textContent = snippet;

    switchScreen('dashboard');
}

// --- v1.3 BATCH 1: HELPER FUNCTIONS ---

function analyzeIntent(title, headings) {
    const t = title.toLowerCase();

    // Commercial / Transactional
    if (t.includes('best') || t.includes('review') || t.includes('top ') || t.includes('vs') || t.includes('price') || t.includes('buy')) {
        return { type: 'Commercial', bg: '#dbeafe', fg: '#2563eb' }; // Blue
    }
    // Informational
    if (t.includes('what') || t.includes('how') || t.includes('guide') || t.includes('tips') || t.includes('why') || t.includes('examples')) {
        return { type: 'Informational', bg: '#d1fae5', fg: '#059669' }; // Green
    }
    // Transactional (Strong)
    if (t.includes('download') || t.includes('coupon') || t.includes('deal')) {
        return { type: 'Transactional', bg: '#fef3c7', fg: '#d97706' }; // Amber
    }

    return { type: 'Mixed / General', bg: '#f1f5f9', fg: '#64748b' };
}

function analyzeEEAT(eeat) {
    let score = 20; // Base score
    const signals = [];

    if (eeat.hasAuthor) { score += 20; signals.push("✔ Author detected"); }
    else signals.push("✖ Missing Author info");

    if (eeat.hasAbout) { score += 20; signals.push("✔ 'About' page linked"); }
    else signals.push("✖ No 'About' page link");

    if (eeat.hasContact) { score += 10; signals.push("✔ Contact info linked"); }

    if (eeat.hasPrivacy) { score += 10; signals.push("✔ Privacy Policy found"); }
    else signals.push("✖ Missing Privacy Policy");

    if (eeat.externalLinks > 0) { score += 20; signals.push(`✔ ${eeat.externalLinks} Outbound citations`); }
    else signals.push("✖ No external citations (E-E-A-T weak)");

    let color = '#ef4444';
    if (score > 70) color = '#10b981';
    else if (score > 40) color = '#f59e0b';

    return { score, color, signals };
}

function checkOptimization(topKeywords, wordCount) {
    const alerts = [];

    // 1. Keyword Stuffing Check
    topKeywords.forEach(k => {
        // Simple density: count / (wordCount/20 approx for tokens) - crude approx, let's use k.score logic if available
        // Better: count / wordCount.
        // Note: k.count is raw freq. wordCount is total.
        const density = (k.count / wordCount) * 100;
        if (density > 2.5) {
            alerts.push(`Over-optimized: "${k.word}" (${density.toFixed(1)}% density)`);
        }
    });

    // 2. Thin Content Warning (Already processed by validation, but Pro might want alert)
    if (wordCount < 600) {
        alerts.push("Content Depth: Thin content (<600 words). Ranking difficulty high.");
    }

    // 3. Semantic Variety (Low LSI)
    // If top 5 keywords account for > 10% of total words (simplified check)
    // Or if total LSI keywords found is very low
    if (topKeywords.length < 5 && wordCount > 300) {
        alerts.push("Low Semantic Variety: Add more LSI keywords.");
    }

    return alerts;
}

// --- v2.0 BATCH 3: SURVIVAL SUITE ---

function analyzeHumanScore(text) {
    // Heuristics for "Human-ness" vs "AI-ness"
    // 1. Personal Pronouns (I, we, my, our) -> High Human
    // 2. Experience Verbs (tested, felt, saw, used) -> High Human
    // 3. Sentence Variance (Burstiness proxy) -> High Human
    // 4. Robotic Transitions (Furthermore, In conclusion, Moreover) -> Low Human (AI)

    const tokens = text.toLowerCase().split(/\s+/);
    const words = tokens.length;
    let humanSignals = 0;
    let aiSignals = 0;

    // A. Personal Pronouns (The "I" Factor)
    const pronouns = ['i', 'me', 'my', 'we', 'our', 'us', 'mine'];
    pronouns.forEach(p => {
        const count = tokens.filter(t => t === p).length;
        humanSignals += count * 2; // Weight x2
    });

    // B. Experience Verbs
    const expVerbs = ['tested', 'tried', 'review', 'used', 'saw', 'felt', 'bought', 'ordered', 'spoke', 'interviewed'];
    expVerbs.forEach(v => {
        const count = tokens.filter(t => t === v).length;
        humanSignals += count * 3; // Weight x3
    });

    // C. Robotic Transitions (AI Markers)
    const robotic = ['furthermore', 'moreover', 'consequently', 'subsequently', 'in conclusion', 'lastly', 'additionally', 'essential to', 'crucial that'];
    robotic.forEach(r => {
        if (text.toLowerCase().includes(r)) aiSignals += 5;
    });

    // D. Burstiness Hack (Variance in sentence length)
    // We can't do full burstiness without complexity, but strict length uniformity is AI-like.
    // If standard deviation of sentence length is high -> Human.
    // Simplifying: Just relying on word choice for now to keep it fast.

    // Score Calculation
    // Base 50. Add Human, Subtract AI.
    let score = 50 + (humanSignals * 2) - (aiSignals * 2);
    score = Math.max(10, Math.min(98, score)); // Clamp

    let msg = "";
    if (score > 80) msg = "✅ High Trust: Strong personal voice & experience markers.";
    else if (score > 50) msg = "⚠️ Moderate: Add more 'I tested/used' phrases.";
    else msg = "🚨 Risky: Sounds robotic/generic. Add personal anecdotes.";

    return { score, msg };
}

function analyzeExperience(text) {
    const signals = [];
    const t = text.toLowerCase();

    // 1. First-hand usage
    if (t.includes('i tested') || t.includes('we tested') || t.includes('our review')) signals.push("✔ 'Tested' Intent Found");
    else signals.push("✖ No 'Tested' Intent");

    // 2. Time-frame
    if (t.match(/\b(in 2024|in 2025|in 2026)\b/)) signals.push("✔ Current Year Reference");
    else signals.push("✖ No Recent Year Mention");

    // 3. Evidence
    if (t.includes('case study') || t.includes('example:')) signals.push("✔ Case Study / Examples");
    else signals.push("✖ Missing Concrete Examples");

    // 4. Evolution
    if (t.includes('before') && t.includes('after')) signals.push("✔ Before/After Comparison");

    return signals;
}

function analyzeFreshness(text) {
    // Current Year trust signal
    const year = new Date().getFullYear();
    const nextYear = year + 1;
    const t = text;

    if (t.includes(year.toString()) || t.includes(nextYear.toString())) return { status: "Fresh", score: 100 };
    if (t.includes((year - 1).toString())) return { status: "Slightly Old", score: 60 };
    return { status: "Outdated?", score: 30 };
}

function analyzeSeoLook(text, headings, primaryKw) {
    if (!primaryKw) return { risk: "Low", msg: "No focus keyword found" };

    // Check if every H2 has the keyword (Over-optimization)
    const h2s = headings.filter(h => h.tag === 'H2');
    if (h2s.length === 0) return { risk: "Low", msg: "Natural heading structure" };

    let spamCount = 0;
    h2s.forEach(h => {
        if (h.text.toLowerCase().includes(primaryKw.toLowerCase())) spamCount++;
    });

    const density = spamCount / h2s.length;
    if (density > 0.7) return { risk: "High", msg: "Warning: Keyword stuffing in H2s." };
    if (density > 0.4) return { risk: "Medium", msg: "Headings look slightly optimized." };
    return { risk: "Low", msg: "Headings look natural." };
}

function analyzeAiOverview(text, headings) {
    // AIO loves: Direct Answer immediately after H2.
    // We verify if we found short paragraphs ~40-60 words.
    // This is a rough proxy.

    let score = 50;
    let msg = "Structure is average.";

    // 1. Structure Check
    if (headings.some(h => h.tag === 'H2' && (h.text.toLowerCase().includes('what') || h.text.toLowerCase().includes('how')))) {
        score += 20;
        msg = "Good Q&A structure found.";
    }

    // 2. Clarity
    // If readability is high (easy), AIO score up
    const read = calculateReadability(text);
    if (read > 60) score += 10;

    return { score: Math.min(95, score), msg };
}

function analyzeScaledRisk(text) {
    // Detects "Mass Produced" signals
    let riskScore = 0;

    // 1. Generic Intros
    if (text.match(/in today'?s (digital|fast-paced) world/i)) riskScore += 30;
    if (text.match(/welcome to the ultimate guide/i)) riskScore += 20;

    // 2. Repetitive Structure (Intro/Outro headers)
    if (text.includes("Conclusion") && text.includes("Introduction")) {
        // Neutral, but standardized.
    }

    // 3. Fluff density
    if (text.match(/look no further/i) || text.match(/we have got you covered/i)) riskScore += 15;

    if (riskScore > 40) return { risk: "High", msg: "Generic 'AI-Scale' patterns detected." };
    if (riskScore > 15) return { risk: "Medium", msg: "Some generic fluff phrases found." };
    return { risk: "Low", msg: "Content feels unique." };
}

function analyzeIntentSatisfaction(text, headings) {
    // Did we answer the user?
    // Proxy: Do H2 questions have content?

    const questions = headings.filter(h => h.tag === 'H2' && h.text.includes('?'));
    if (questions.length === 0) return { score: 50, msg: "No question-based headings found." };

    // If we have questions, assume good intent alignment. 
    // Real depth check is hard without DOM position.

    return { score: 85, msg: "Questions in headings suggest good intent coverage." };
}

function analyzeVolatility(freshness, authorityScore) {
    // If content is old AND authority is low -> Vulnerable
    // If content is Fresh AND authority is High -> Immune

    let score = 50;
    if (freshness.status === 'Fresh') score += 20;
    if (authorityScore > 60) score += 20;

    if (score > 80) return "High Immunity 🛡️";
    if (score > 50) return "Stable";
    return "Vulnerable 📉";
}

// --- v1.3 BATCH 2: CONTENT TOOLS ---

function analyzeSnippetOpp(intentType, headings, title) {
    const hText = headings.map(h => h.text.toLowerCase());
    const t = title.toLowerCase();

    // 1. Definition Snippet (Paragraph)
    if (intentType === 'Informational') {
        if (t.includes('what is') || t.includes('meaning')) {
            return "🔥 High Chance: Definition Snippet (Add direct answer in <p> tag)";
        }
        if (hText.some(h => h.includes('what is') || h.includes('definition'))) {
            return "✅ Good: Definition Snippet Opportunity detected.";
        }
    }

    // 2. List Snippet
    if (t.includes('best') || t.includes('top') || t.includes('steps')) {
        return "🔥 High Chance: List Snippet (Ensure H2s or <ol> used)";
    }

    // 3. Table Snippet
    if (t.includes('vs') || t.includes('comparison') || t.includes('price') || t.includes('specs')) {
        return "🔥 High Chance: Comparison Table Snippet";
    }

    // Default
    return "ℹ️ Optimize for Long-form Content (Depth)";
}

function generateProOutline(data) {
    const intent = analyzeIntent(data.title, data.headings).type;
    const topic = data.title;
    const lsi = lsiKeywords.slice(0, 5).map(k => k.word).join(', ');

    let template = `# AI Content Outline for: ${topic}\n`;
    template += `**Intent:** ${intent} | **Focus:** ${lsi}\n\n`;

    if (intent === 'Commercial') {
        template += `## Introduction\n- Hook: Why ${topic} matters.\n- Intent: "Best options for 2024".\n\n`;
        template += `## Top Picks (Comparison)\n- Product 1 (Best Overall)\n- Product 2 (Best Budget)\n- Product 3 (Best Premium)\n\n`;
        template += `## Buying Guide\n- Factor 1: Price\n- Factor 2: Features to Look for\n\n`;
        template += `## FAQ\n- Is ${topic} worth it?\n- How much does it cost?\n\n`;
    } else if (intent === 'Informational') {
        template += `## Introduction\n- Define: What is ${topic}?\n- Why important?\n\n`;
        template += `## How it Works / Core Concepts\n- Step 1 / Detail 1\n- Step 2 / Detail 2\n\n`;
        template += `## Benefits & Examples\n- Key Benefit 1\n- Real world example\n\n`;
        template += `## Conclusion\n- Summary & Next Steps\n\n`;
    } else {
        template += `## Introduction\n- Overview of ${topic}\n\n`;
        template += `## Main Section 1\n\n## Main Section 2\n\n## Conclusion\n\n`;
    }

    template += `> **SEO Tip:** Ensure you answer "People Also Ask" questions related to ${topic}.`;
    return template;
}

function exportCsv() {
    if (!pageData) return;

    let csvContent = "\uFEFF"; // BOM for Excel
    csvContent += "Type,Keyword,Count,Score\n";

    // Primary
    csvContent += `Topic,${cleanCSV(pageData.title)},1,100\n`;

    // NLP
    nlpKeywords.forEach(k => {
        csvContent += `NLP,${cleanCSV(k.word)},${k.count},${k.score.toFixed(2)}\n`;
    });

    // LSI
    lsiKeywords.forEach(k => {
        csvContent += `LSI,${cleanCSV(k.word)},${k.count},${(k.count * 10).toFixed(0)}\n`;
    });

    const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement("a");
    const url = URL.createObjectURL(blob);
    link.setAttribute("href", url);
    link.setAttribute("download", "seo_data.csv");
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
}

function cleanCSV(str) {
    if (!str) return "";
    return `"${str.replace(/"/g, '""')}"`;
}


// --- Prompt Engine ---

function extractNLPKeywords(text, headings, titleTokens) {
    // NLP Goal: Medium/Long tail phrases (2-4 words) that are RELEVANT to the Title/Topic
    const tokens = cleanAndTokenize(text);
    if (tokens.length < 10) return [];

    const candidates = {};

    // Helper to process n-grams
    const processNgrams = (n) => {
        for (let i = 0; i < tokens.length - n + 1; i++) {
            const slice = tokens.slice(i, i + n);
            const phrase = slice.join(' ');

            // Base score = 1
            let score = 1;

            // Relevancy Boost: Does it contain a title word?
            const hasTitleWord = slice.some(word => titleTokens.includes(word));
            if (hasTitleWord) score += 5;

            candidates[phrase] = (candidates[phrase] || 0) + score;
        }
    };

    processNgrams(2);
    processNgrams(3);
    processNgrams(4);

    return Object.keys(candidates)
        .map(w => ({ word: w, count: 1, score: candidates[w] }))
        .filter(item => item.score > 2)
        .sort((a, b) => b.score - a.score);
}

function extractLSIKeywords(tokens) {
    // LSI Goal: Frequent phrases in the body
    const ngrams = {};

    const addNgram = (n) => {
        for (let i = 0; i < tokens.length - n + 1; i++) {
            const phrase = tokens.slice(i, i + n).join(' ');
            ngrams[phrase] = (ngrams[phrase] || 0) + 1;
        }
    };

    addNgram(2);
    addNgram(3);
    addNgram(4);

    return Object.keys(ngrams)
        .filter(phrase => ngrams[phrase] >= 1)
        .map(phrase => ({ word: phrase, count: ngrams[phrase], score: ngrams[phrase] }))
        .sort((a, b) => b.score - a.score);
}

// Removing redundant subsets (e.g. if "best seo tips" is kept, drop "seo tips")
function removeRedundantPhrases(list) {
    // Assume list is sorted by Score DESC
    const result = [];

    for (const item of list) {
        // Is this phrase a subset of an already picked phrase?
        // e.g. item="seo tips", picked="best seo tips" -> item is subset.
        const isSubset = result.some(picked => picked.word.includes(item.word));

        // Is an already picked phrase a subset of this new, lower-scored one? (Less likely since we sort by score, but possible)
        // If "seo" (score 100) vs "seo strategy" (score 10). We probably want "seo strategy" if we are looking for long tail, 
        // OR we want "seo" if we want core topics. 
        // User asked for "Similar keywords repeat nh ho".
        // Let's go strict: If it's a substring of another, drop it.

        if (!isSubset) {
            result.push(item);
        }
    }
    return result;
}

function cleanAndTokenize(text) {
    if (!text) return [];
    // robustness: if replace fails, we still want something
    try {
        return text.toLowerCase()
            .replace(/[^\w\s]/g, '') // simpler regex
            .split(/\s+/)
            .filter(w => w.length > 2 && !isStopWord(w) && !/^\d+$/.test(w));
    } catch (e) {
        return text.toLowerCase().split(/\s+/);
    }
}

// --- Advanced Logic Helpers ---

function calculateReadability(text) {
    // Approximate Flesch-Kincaid
    const sentences = text.split(/[.!?]+/).length || 1;
    const words = text.split(/\s+/).length || 1;
    const syllables = text.length / 3; // Rough approximation

    const score = 206.835 - (1.015 * (words / sentences)) - (84.6 * (syllables / words));
    return Math.max(0, Math.min(100, Math.round(score)));
}

function calculateSentiment(text) {
    // Ultra-simple bag-of-words
    const positives = ['good', 'great', 'excellent', 'amazing', 'best', 'benefit', 'win', 'success', 'improve', 'easy', 'love', 'top', 'pro'];
    const negatives = ['bad', 'worst', 'fail', 'hard', 'difficult', 'loss', 'avoid', 'stop', 'wrong', 'error', 'hate', 'risk', 'issue'];

    const tokens = cleanAndTokenize(text);
    let score = 0;

    tokens.forEach(t => {
        if (positives.includes(t)) score++;
        if (negatives.includes(t)) score--;
    });

    if (score > 2) return { label: "Positive", color: "#10b981" };
    if (score < -2) return { label: "Negative", color: "#ef4444" };
    return { label: "Neutral", color: "#64748b" };
}

function cleanAndTokenize(text) {
    if (!text) return [];

    // Strict Cleanup for NLP/LSI: 
    // 1. Lowercase
    // 2. Remove all non-letter characters (replace digits/punctuation with space)
    // 3. Split by space
    // 4. Filter empty, short words, stopwords using the expanded list

    // Regex explanation: [^a-z\s] matches anything that is NOT a lowercase letter or whitespace.
    // Replacing it with ' ' ensures that "1200x800" becomes "x" (removed) and "500px" becomes "px" (removed).
    const pureText = text.toLowerCase().replace(/[^a-z\s]/g, ' ');

    return pureText.split(/\s+/)
        .filter(w => w.length > 2 && !isStopWord(w));
}

function isStopWord(w) {
    const stops = [
        // Standard
        'the', 'be', 'to', 'of', 'and', 'a', 'in', 'that', 'have', 'i', 'it', 'for', 'not', 'on', 'with', 'he', 'as', 'you', 'do', 'at',
        'this', 'but', 'his', 'by', 'from', 'they', 'we', 'say', 'her', 'she', 'or', 'an', 'will', 'my', 'one', 'all', 'would', 'there',
        'their', 'what', 'so', 'up', 'out', 'if', 'about', 'who', 'get', 'which', 'go', 'me', 'when', 'make', 'can', 'like', 'time', 'no',
        'just', 'him', 'know', 'take', 'people', 'into', 'year', 'your', 'good', 'some', 'could', 'them', 'see', 'other', 'than', 'then',
        'now', 'look', 'only', 'come', 'its', 'over', 'think', 'also', 'back', 'after', 'use', 'two', 'how', 'our', 'work', 'first', 'well',
        'way', 'even', 'new', 'want', 'because', 'any', 'these', 'give', 'day', 'most', 'us', 'are', 'is', 'was', 'were',

        // Image/Tech noise specific to user issue
        'image', 'img', 'src', 'alt', 'width', 'height', 'px', 'jpg', 'jpeg', 'png', 'gif', 'webp', 'size', 'max', 'min', 'style',
        'class', 'div', 'span', 'https', 'http', 'com', 'www', 'content', 'load', 'loading', 'data', 'lazy'
    ];
    return stops.includes(w);
}

function formatNumber(num) {
    return num > 999 ? (num / 1000).toFixed(1) + 'k' : num;
}

function showError(msg) {
    displays.error.textContent = msg;
    displays.error.style.display = 'block';
}

function displayOverviewMetrics(metrics) {
    displays.words.textContent = formatNumber(metrics.words);
    displays.readability.textContent = metrics.readability;
    displays.time.textContent = metrics.time;
    displays.sentiment.textContent = metrics.sentiment.label;
    displays.sentiment.style.color = metrics.sentiment.color;
}

function calculateKeywordDensity(text) {
    const tokens = cleanAndTokenize(text);
    const counts = {};
    tokens.forEach(t => counts[t] = (counts[t] || 0) + 1);

    // Sort
    return Object.keys(counts)
        .map(w => ({ word: w, count: counts[w], score: counts[w] / tokens.length }))
        .sort((a, b) => b.count - a.count)
        .slice(0, 10);
}

function renderDensityChart(densityItems) {
    const list = displays.density;
    if (!list) return;

    if (densityItems.length === 0) {
        list.innerHTML = `<div style="padding:10px; color:#64748b; font-size:11px;">No data available.</div>`;
        return;
    }

    const max = densityItems[0].count;
    list.innerHTML = densityItems.map(item => `
        <div class="density-row">
            <span style="width:80px; font-size:11px; overflow:hidden; text-overflow:ellipsis;">${item.word}</span>
            <div class="progress-bg">
                <div class="progress-fill" style="width:${(item.count / max) * 100}%"></div>
            </div>
            <span style="font-size:10px; width:30px; text-align:right;">${item.count}</span>
        </div>
    `).join('');
}

// --- Prompt Engine ---

// --- Prompt Engine ---

function updateInternalPrompt() {
    const data = pageData || { title: inputs.topic.value, headings: [] }; // Fallback
    const template = inputs.template.value;
    let prompt = "";

    if (template === 'seo') {
        prompt = `Write an SEO optimized article about "${data.title}". Target the keyword "${inputs.primary.value}". Include these headings: \n${data.headings.map(h => h.text).join('\n')}`;
    } else if (template === 'social') {
        prompt = `Write a viral LinkedIn post about "${data.title}". Use a hook in the first sentence.`;
    } else if (template === 'summary') {
        prompt = `Summarize the following content in 3 bullet points:\n\n${data.rawText ? data.rawText.substring(0, 2000) : '(No content extracted)'}`;
    } else if (template === 'rewrite') {
        prompt = `Rewrite the following H1 for better CTR, targeting "${inputs.primary.value}":\n"${data.title}"`;
    } else if (template === 'outline') {
        prompt = generateProOutline(data);
    }

    document.getElementById('finalOutput').value = prompt;
}

function saveAndOpenGenerator() {
    // Pro Security Check
    if (typeof AuthService === 'undefined' || !AuthService.isPro) {
        alert("🔒 Pro Feature: Please upgrade to access the AI Prompt Generator.");
        return;
    }

    const topic = inputs.topic.value;
    const nlp = document.getElementById('inputNLPKw').value;

    // Merge Local LSI with AI LSI if available
    let lsi = inputs.secondary.value;
    if (typeof aiGeneratedResults !== 'undefined' && aiGeneratedResults.keywords && aiGeneratedResults.keywords.length > 0) {
        if (lsi) lsi += ", ";
        lsi += aiGeneratedResults.keywords.join(', ');
    }

    // Save data to storage
    chrome.storage.local.set({
        seoData: {
            title: topic,
            nlp: nlp,
            lsi: lsi,
            headings: pageData ? pageData.headings : []
        }
    }, () => {
        // Open the Prompt Generator Page
        chrome.tabs.create({ url: 'prompt_generator.html' });
    });
}


function copyOutput() {
    displays.output.select();
    document.execCommand('copy');
    const btn = document.getElementById('btnCopy');
    btn.textContent = "Copied!";
    setTimeout(() => btn.textContent = "Copy Prompt", 2000);
}

function checkDomainAge() {
    chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
        if (tabs[0]?.url) {
            try {
                const url = new URL(tabs[0].url);
                const domain = url.hostname;
                chrome.tabs.create({ url: `https://who.is/whois/${domain}` });
            } catch (e) {
                alert("Invalid URL for domain check.");
            }
        }
    });
}



// --- v2.2 AI LOGIC ---

function runAiAnalysis() {
    // 1. Pro Check
    if (typeof AuthService === 'undefined' || !AuthService.isPro) {
        alert("🔒 Multi-Trillion Dollar AI Brain Reserved for Pro Users! Upgrade to unlock.");
        return;
    }

    // 2. UI State
    const list = document.getElementById('aiKeywordList');
    const btnRegen = document.getElementById('btnAiKeywords');
    const btnActivate = document.getElementById('btnAiBrainActivate');

    // UI Feedback (Toggle both to show activity)
    if (btnRegen) {
        btnRegen.disabled = true;
        btnRegen.textContent = "Processing...";
    }
    if (btnActivate) {
        btnActivate.disabled = true;
        btnActivate.textContent = "Thinking... 🧠";
    }

    document.getElementById('inputNLPKw').placeholder = "AI is thinking...";

    // 3. Call Service
    const context = {
        title: pageData ? pageData.title : "Untitled",
        headings: pageData ? pageData.headings : []
    };

    // Advanced Prompt for Full Analysis
    const prompt = `
    Analyze this content context and provide:
    1. 1 Primary "Focus Keyword" (The main topic).
    2. 20 Semantic/LSI Keywords.
    3. 20 NLP/Contextual Keywords.
    4. Search Intent (Informational, Commercial, Transactional, Navigational).
    5. Sentiment (Positive, Negative, Neutral).

    Title: "${context.title}"
    Headings: "${context.headings.map(h => h.text).join('; ')}"

    Return ONLY a JSON object with this structure:
    {
      "focus_keyword": "String",
      "lsi_keywords": ["..."],
      "nlp_keywords": ["..."],
      "intent": "String",
      "sentiment": "String"
    }
    `;

    geminiService.generateContent({ prompt: prompt, forceJson: true }, (response) => {
        // Reset Buttons
        if (btnRegen) {
            btnRegen.disabled = false;
            btnRegen.textContent = "✨ Regenerate";
        }
        if (btnActivate) {
            btnActivate.disabled = false;
            btnActivate.textContent = "Activate";
        }

        document.getElementById('inputNLPKw').placeholder = "AI will extract semantic terms here...";

        if (response.error) {
            alert(`Error: ${response.error}`);
            // Also reset to error state if needed
            return;
        }

        try {
            const resultPayload = response.result;
            // Handle if result is already an object (due to forceJson) or string
            let data;
            if (typeof resultPayload === 'object') {
                data = resultPayload;
            } else {
                const cleanJson = resultPayload.replace(/```json/g, '').replace(/```/g, '').trim();
                data = JSON.parse(cleanJson);
            }

            // Store results globally
            aiGeneratedResults = {
                keywords: [...(data.lsi_keywords || []), ...(data.nlp_keywords || [])],
                intent: data.intent,
                sentiment: data.sentiment
            };

            // 1. Populate UI
            if (data.nlp_keywords) document.getElementById('inputNLPKw').value = data.nlp_keywords.join(', ');
            if (data.lsi_keywords) document.getElementById('inputSecondaryKw').value = data.lsi_keywords.join(', ');
            if (data.focus_keyword) document.getElementById('inputPrimaryKw').value = data.focus_keyword; // Auto-Extract Focus

            // 2. Update Metrics UI (Intent & Sentiment)
            if (data.intent) {
                const intentBadge = document.getElementById('intentBadge');
                intentBadge.textContent = "AI: " + data.intent;
                intentBadge.style.background = "#dbeafe";
                intentBadge.style.color = "#1e40af";
            }

            if (data.sentiment) {
                const sentMetric = document.getElementById('metricSentiment');
                sentMetric.textContent = data.sentiment;
                if (data.sentiment.toLowerCase().includes('positive')) sentMetric.style.color = "#10b981";
                else if (data.sentiment.toLowerCase().includes('negative')) sentMetric.style.color = "#ef4444";
                else sentMetric.style.color = "#64748b";
            }

        } catch (e) {
            console.error("AI Parse Error", e);
            alert("Error parsing AI response. Try again.");
        }
    });
}

function runAiEeatAnalysis() {
    if (typeof AuthService === 'undefined' || !AuthService.isPro) {
        alert("🔒 Pro Feature. Upgrade to access.");
        return;
    }

    const btn = document.getElementById('btnAiEeat');
    const container = document.getElementById('eeatSignals');
    const scoreVal = document.getElementById('eeatScore');

    btn.disabled = true;
    btn.textContent = "Scanning...";
    container.innerHTML = "Consulting AI Expert...";

    const context = {
        title: pageData.title,
        headings: pageData.headings,
        textSample: pageData.rawText ? pageData.rawText.substring(0, 1500) : "No content text found."
    };

    const prompt = `
    Act as a Google Quality Rater. Analyze this content for E-E-A-T (Experience, Expertise, Authoritativeness, Trustworthiness).
    
    Title: "${context.title}"
    Content Snippet: "${context.textSample}"

    Provide:
    1. A Score (0-100).
    2. 3 Specific Strengths (start with +).
    3. 3 Specific Weaknesses (start with -).
    
    Return strict JSON:
    {
      "score": 85,
      "analysis": ["+ Strength 1...", "+ Strength 2...", "- Weakness 1..."]
    }
    `;

    geminiService.generateContent(prompt, (response) => {
        btn.disabled = false;
        btn.textContent = "⚡ AI Deep Scan";

        if (response.error) {
            container.innerHTML = "Analysis failed.";
            return;
        }

        try {
            const cleanJson = response.result.replace(/```json/g, '').replace(/```/g, '').trim();
            const data = JSON.parse(cleanJson);

            scoreVal.textContent = data.score + "/100";
            // Color code
            if (data.score > 80) scoreVal.style.color = "#10b981";
            else if (data.score > 50) scoreVal.style.color = "#f59e0b";
            else scoreVal.style.color = "#ef4444";

            container.innerHTML = data.analysis.map(line => {
                const color = line.startsWith('+') ? '#065f46' : '#991b1b';
                return `<div style="color:${color}; margin-bottom:2px;">${line}</div>`;
            }).join('');

        } catch (e) {
            container.innerHTML = "Error parsing AI E-E-A-T data.";
        }
    });
}

function downloadReport() {
    if (!pageData) return;

    const topic = inputs.topic.value;
    const nlp = document.getElementById('inputNLPKw').value; // Get from inputs
    const lsi = inputs.secondary.value; // Get from inputs

    // Construct CSV Content
    let csv = [];
    csv.push(`"SEO REPORT: ${topic.replace(/"/g, '""')}"`);
    csv.push(`"Date","${new Date().toLocaleDateString()}"`);
    csv.push(""); // Spacer

    // Metrics
    csv.push(`"METRICS"`);
    csv.push(`"Word Count","${displays.words.textContent}"`);
    csv.push(`"Readability","${displays.readability.textContent}"`);
    csv.push(`"Reading Time","${displays.time.textContent}"`);
    csv.push(`"Sentiment","${displays.sentiment.textContent}"`);
    csv.push("");

    // Keywords
    csv.push(`"KEYWORDS"`);
    csv.push(`"Primary Keyword","${inputs.primary.value.replace(/"/g, '""')}"`);
    csv.push(`"NLP Keywords (Contextual)","${nlp.replace(/"/g, '""')}"`);
    csv.push(`"LSI Keywords (Phrases)","${lsi.replace(/"/g, '""')}"`);
    csv.push("");

    // Headings
    csv.push(`"HEADINGS STRUCTURE"`);
    csv.push(`"Tag","Text"`);
    pageData.headings.forEach(h => {
        csv.push(`"${h.tag}","${h.text.replace(/"/g, '""')}"`);
    });
    csv.push("");

    // Generated Prompt (if any)
    const promptVal = displays.output.value;
    if (promptVal) {
        csv.push(`"GENERATED PROMPT"`);
        csv.push(`"${promptVal.replace(/"/g, '""')}"`);
    }

    // Download Logic
    const blob = new Blob([csv.join("\n")], { type: 'text/csv;charset=utf-8;' });
    const url = URL.createObjectURL(blob);
    const link = document.createElement("a");
    link.setAttribute("href", url);
    link.setAttribute("download", `seo-report-${topic.substring(0, 20).replace(/[^a-z0-9]/gi, '_').toLowerCase()}.csv`);
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
    link.click();
    document.body.removeChild(link);
}

// End of file

// Add helper to open AI Prompt Page
function secureOpenAiGenerator() {
    // Pro Security Check
    if (typeof AuthService === 'undefined' || !AuthService.isPro) {
        alert("🔒 Pro Feature: Please upgrade to access the AI Prompt Generator (Auto-Fill).");
        return;
    }

    // 1. Gather Data (Robustly from UI inputs to capture user edits)
    const topic = document.getElementById('inputTopic').value ||
        (pageData ? pageData.title : "") ||
        "Untitled Article";

    const nlp = document.getElementById('inputNLPKw').value || "";
    const lsi = document.getElementById('inputSecondaryKw').value || "";

    // 2. Prepare Data Package
    // We pass the raw headings array so the receiver can format it nicely
    const headings = (pageData && pageData.headings) ? pageData.headings : [];

    const dataToSave = {
        title: topic,
        nlp: nlp,
        lsi: lsi,
        headings: headings
    };

    console.log("Saving SEO Data for AI Gen:", dataToSave);

    // 3. Save & Redirect
    chrome.storage.local.set({
        seoData: dataToSave
    }, () => {
        // Open the AI Prompt Generator Page
        chrome.tabs.create({ url: 'ai_prompt_generator.html' });
    });
}

// --- PRO GATING LOGIC ---

function openWhatsApp(message) {
    const phoneNumber = "923226723073";
    const text = encodeURIComponent(message || "Hello, I want to upgrade to SEO Extractor Pro.");
    const url = `https://wa.me/${phoneNumber}?text=${text}`;
    window.open(url, '_blank');
}

function updateUILocks() {
    const btn = document.getElementById('btnLoginAction'); // Header Button (Legacy)
    const lockOverlay = document.getElementById('proLockOverlay');
    const promptBtn = document.getElementById('btnGoToPrompt');

    const user = (typeof AuthService !== 'undefined') ? AuthService.user : null;
    const isPro = (typeof AuthService !== 'undefined') ? AuthService.isPro : false;

    // 1. Header Button Text (Legacy: handled by setupHeaderUser largely, but we can refine here)
    // Actually, setupHeaderUser handles the Login vs User display. 
    // updateUILocks should focus on PRO gating features.

    // We can call setupHeaderUser here to ensure sync
    setupHeaderUser();

    // 2. Feature Gating
    if (isPro) {
        // PRO MODE
        if (lockOverlay) lockOverlay.style.display = 'none';

        if (promptBtn) {
            promptBtn.textContent = "Open Prompt Generator";
            promptBtn.disabled = false;
            promptBtn.style.opacity = "1";
            promptBtn.style.cursor = "pointer";
        }
    } else {
        // FREE MODE
        if (lockOverlay) {
            lockOverlay.style.display = 'flex';
            // ... (rest of lock overlay logic same)
        }
    }

    // ... (rest of the function)

    // 1. API Key Settings
    const inputKey = document.getElementById('inputGeminiKey');
    const btnSave = document.getElementById('btnSaveKey');

    if (inputKey && btnSave) {
        if (!isPro) {
            inputKey.disabled = true;
            inputKey.placeholder = "🔒 Upgrade to Pro to customize AI";
            btnSave.disabled = true;
            btnSave.textContent = "Pro Only";
            btnSave.style.background = "#94a3b8";
            btnSave.style.cursor = "not-allowed";
        } else {
            inputKey.disabled = false;
            inputKey.placeholder = "Paste your API key here...";
            btnSave.disabled = false;
            btnSave.textContent = "Save Settings";
            btnSave.style.background = "#2563eb";
            btnSave.style.cursor = "pointer";
        }
    }

    // 2. Helper Buttons (AI Gen & Brain)
    const btnAi = document.getElementById('btnAiKeywords');
    const btnBrain = document.getElementById('btnAiBrainActivate');

    // Function to lock a button
    const lockButton = (btn, text) => {
        if (!btn) return;
        btn.innerHTML = text || "🔒 Pro Only";
        btn.style.opacity = "0.7";
        btn.style.cursor = "pointer"; // Upsell click

        const clone = btn.cloneNode(true);
        btn.parentNode.replaceChild(clone, btn);
        clone.addEventListener('click', () => openWhatsApp("I want to upgrade to unlock AI Keywords & Brain."));
    };

    // Function to unlock a button (Restore original state/listener)
    // Note: Best handled by page reload, but we can reset styles
    const unlockButton = (btn, originalText) => {
        if (!btn) return;
        btn.innerHTML = originalText;
        btn.style.opacity = "1";
        btn.style.cursor = "pointer";
        // Re-attach listener is tricky without named function or explicit re-bind.
        // We rely on reload mostly, but if dynamic:
        const clone = btn.cloneNode(true);
        btn.parentNode.replaceChild(clone, btn);
        clone.addEventListener('click', runAiAnalysis);
    };

    if (!isPro) {
        lockButton(btnAi, "🔒 Pro");
        lockButton(btnBrain, "🔒 Activate");
    } else {
        // Ensure they are unlocked (mainly for dynamic login)
        unlockButton(btnAi, "✨ Regenerate");
        unlockButton(btnBrain, "Activate");
    }

    // 3. Tab Visuals
    const keywordTabBtn = document.querySelector('.tab-btn[data-tab="tab-keywords"]');
    if (keywordTabBtn) {
        if (!isPro) {
            if (!keywordTabBtn.innerHTML.includes('🔒')) {
                keywordTabBtn.innerHTML += ` <span style="font-size:10px">🔒</span>`;
            }
            keywordTabBtn.style.opacity = "0.7";
            keywordTabBtn.title = "Upgrade to unlock";
        } else {
            keywordTabBtn.innerHTML = "Keywords";
            keywordTabBtn.style.opacity = "1";
            keywordTabBtn.title = "";
        }
    }

    // 4. AI Prompt Gen Button (Auto-Fill) - VISUAL LOCK
    const btnAiPrompt = document.getElementById('btnGoToAiPrompt');
    if (btnAiPrompt) {
        if (!isPro) {
            // LOCK IT
            btnAiPrompt.innerHTML = `
                <svg width="18" height="18" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 15v2m-6 4h12a2 2 0 002-2v-6a2 2 0 00-2-2H6a2 2 0 00-2 2v6a2 2 0 002 2zm10-10V7a4 4 0 00-8 0v4h8z" />
                </svg>
                AI Auto-Fill (Pro Only)
            `;
            btnAiPrompt.style.background = "#f1f5f9"; // Grey
            btnAiPrompt.style.color = "#94a3b8";
            btnAiPrompt.style.borderColor = "#e2e8f0";
            btnAiPrompt.style.cursor = "pointer"; // Still clickable for upsell

            // Clone to remove old listeners and add upsell listener
            const clone = btnAiPrompt.cloneNode(true);
            btnAiPrompt.parentNode.replaceChild(clone, btnAiPrompt);
            clone.addEventListener('click', () => openWhatsApp("I want to upgrade to Pro for AI Auto-Fill Prompt Gen."));
        } else {
            // UNLOCK IT (Restore original if needed, but usually page refresh handles state)
            btnAiPrompt.innerHTML = `
                <svg width="18" height="18" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 10V3L4 14h7v7l9-11h-7z" />
                </svg>
                AI Prompt Gen (Auto-Fill)
            `;
            btnAiPrompt.style.background = "#f0fdf4";
            btnAiPrompt.style.color = "#15803d";
            btnAiPrompt.style.borderColor = "#86efac";
            btnAiPrompt.style.cursor = "pointer";

            // Re-attach original handler (since we might have cloned it previously)
            const clone = btnAiPrompt.cloneNode(true);
            btnAiPrompt.parentNode.replaceChild(clone, btnAiPrompt);
            clone.addEventListener('click', secureOpenAiGenerator);
        }
    }
}

// Hook into Tab Clicks to enforce lock (Capture Phase)
document.querySelectorAll('.tab-btn').forEach(btn => {
    btn.addEventListener('click', (e) => {
        const tab = btn.getAttribute('data-tab');
        if (tab === 'tab-keywords') {
            if (typeof AuthService !== 'undefined' && !AuthService.isPro) {
                e.stopImmediatePropagation();
                e.preventDefault();

                const confirmUnlock = confirm("🔒 This feature is for Pro users.\n\nClick OK to upgrade via WhatsApp!");
                if (confirmUnlock) {
                    openWhatsApp(`I want to unlock the Keywords feature.`);
                }
                return false;
            }
        }
    }, true);
});

// Initial check on load
document.addEventListener('DOMContentLoaded', () => {
    // Single source of truth for initialization
    setupAuth();
    setTimeout(updateUILocks, 500);

    // FORCE LOGIN ACTION (Brute Force Fix)
    const forceLoginBtn = document.getElementById('btnLoginAction');
    const btnWebLogin = document.getElementById('btnWebLogin'); // Assuming btnWebLogin is a new element or intended to be used
    if (btnWebLogin) {
        btnWebLogin.addEventListener('click', () => {
            window.open('https://app.survivalenginepro.com/', '_blank');
        });
    }
    if (forceLoginBtn) { // Original forceLoginBtn logic
        forceLoginBtn.addEventListener('click', () => {
            const ov = document.getElementById('loginOverlay');
            if (ov) {
                ov.style.display = 'flex';
                // Also trigger logic if needed
                if (typeof setMode === 'function') setMode('login');
            }
        });
    }
});
