
// ai_prompt_generator.js - Re-Integrated with robust GeminiService

const geminiService = new GeminiService();
let currentSeoData = {}; // Global storage for prompt context
console.log("AI Prompt Generator Script Loaded (v3.0 - Unified Service)");

document.addEventListener('DOMContentLoaded', () => {
    // Load SEO Data
    chrome.storage.local.get(['seoData'], (result) => {
        if (result.seoData) populateForm(result.seoData);
    });

    // Bind Actions
    document.getElementById('btnAiGeneratePrompt').addEventListener('click', generateMetaPrompt);
    document.getElementById('btnCopy').addEventListener('click', copyToClipboard);
});

function populateForm(data) {
    if (!data) return;
    currentSeoData = data; // Store for global access
    document.getElementById('inputTitle').value = data.title || '';

    // Convert array of objects to string for display
    const headText = Array.isArray(data.headings) ?
        data.headings.map(h => `${h.tag}: ${h.text}`).join('\n') : '';

    document.getElementById('inputHeadings').value = headText;
    document.getElementById('inputNLP').value = data.nlp || '';
    document.getElementById('inputLSI').value = data.lsi || '';
}

function generateMetaPrompt() {
    const btn = document.getElementById('btnAiGeneratePrompt');
    const outputArea = document.getElementById('finalOutput');
    const outputSection = document.getElementById('outputSection');

    // UI Reset
    btn.disabled = true;
    btn.textContent = "🤖 Connecting to Brain...";
    outputSection.style.display = 'block';
    outputSection.scrollIntoView({ behavior: 'smooth' });
    outputArea.value = "Analyzing requirements...\n";

    // Gather Inputs
    const title = document.getElementById('inputTitle').value || "Topic";
    const tone = document.getElementById('inputTone').value || "Professional";
    const headings = document.getElementById('inputHeadings').value || "";
    const nlp = document.getElementById('inputNLP').value || "";
    const lsi = document.getElementById('inputLSI').value || "";

    // Construct Context
    const systemPrompt = `
    You are an expert SEO Prompt Engineer specializing in "Google Core Update Survival" and "Google AI Overviews (SGE)".

    I need a "Mega-Prompt" to write a high-ranking blog post about: "${title}".
    
    ## SEO DATA:
    - Target Tone: ${tone}
    - Primary Keywords (NLP): ${nlp}
    - Secondary Keywords (LSI): ${lsi}
    - Search Intent: ${currentSeoData.intent || "Not Specified"}
    - Sentiment Analysis: ${currentSeoData.sentiment || "Neutral"}
    - Structure:
    ${headings}

    ## CRITICAL REQUIREMENTS FOR THE PROMPT YOU GENERATE:
    The prompt you write must instruct the AI to follow these exact rules:

    1. **E-E-A-T & First Person Experience**:
       - The article MUST be written in the **First Person (I, Me, My)** to demonstrate real-world experience.
       - Include personal anecdotes or specific "I tested this" statements to satisfy Google's Experience signal.

    2. **Survival & Technical Strategy**:
       - The content must be "Helpful Content" compliant—focus on solving the user's problem, not just SEO.
       - Use short, punchy sentences (Burstiness) and varied sentence structures (Perplexity) to bypass AI detection.
       - Ensure technical accuracy for the specific niche.

    3. **The "Perfect Introduction" Rule (MANDATORY)**:
       - **Perspective**: STRICTLY **First Person ONLY** ("I", "Me", "My"). Do NOT use "You", "We", or Third Person in the introduction.
       - **Total Length**: ~110 words.
       - **Structure**: Exactly **two paragraphs**.
       - **Paragraph 1 (45-55 words)**: Hook the reader by sharing a personal struggle/experience ("I found...", "I struggled with...") to build an **Emotional Connection**, AND include **5-6 LSI Keywords** naturally.
       - **Paragraph 2 (45-55 words)**: State a clear **Promise** of what "I will show you" (implied you is okay, but focus on I will demonstrate), AND include another **5-6 LSI Keywords**.
       - **Style**: SGE-style (Direct, Answer-First).

    4. **Content Body Structure (Strict Rules)**:
       - **Total Article Length**: Must exceed **1000 words**.
       - **Per Heading**: Write exactly **two paragraphs**.
       - **Per Paragraph**: Length must be **45-55 words**.
       - **Keyword Density**: MANDATORY usage of **5-6 Semantic/NLP keywords** per paragraph.
       - **Formatting**: Do **NOT** bold the keywords. Keep them plain text.
       - **Visuals**: Use **Tables** where appropriate (e.g., for comparisons, data lists, or easy-to-read summaries).
       - **Flow**: Start each section with strong topic sentences.

    5. **Tone & Style (Human & Personal)**:
       - **Tone**: ${tone}
       - **Primary Perspective**: Dominant use of **First Person ("I", "Me", "My")** to show authority and authentic experience.
       - **Secondary Perspective**: Use Second Person ("You") and Third Person **MINIMALLY** (keep them silent/minor).
       - **Human Touch**: Avoid robotic phrasing or repetitive transitions. Use contractions (e.g., "don't" instead of "do not"), natural conversational flow, and idioms where appropriate.
       - **Sentence Structure**: High Burstiness (varied sentence lengths) and High Perplexity (unpredictable vocabulary) to mimic human writing.

    **TASK**: Write the exact, ready-to-copy PROMPT I should give to an AI (like ChatGPT/Gemini/Claude) to generate this full article.
    OUTPUT ONLY THE PROMPT.
    `;

    // Use The Unified Service (Handles Model Selection & Fallback)
    outputArea.value += "Dispatching to optimal AI Model...\n";

    geminiService.generateContent({
        prompt: systemPrompt,
        forceJson: false // We want text for the prompt
    }, (response) => {
        btn.disabled = false;
        btn.textContent = "✨ Generate Perfect Prompt";

        if (response.error) {
            outputArea.value += `\n❌ Error: ${response.error}`;
            if (response.error.includes('Key')) {
                alert("Valid API Key required. Please check Extension Settings.");
            }
        } else {
            outputArea.value = response.result;
        }
    });
}

function copyToClipboard() {
    const output = document.getElementById('finalOutput');
    output.select();
    document.execCommand('copy');
    const btn = document.getElementById('btnCopy');
    const old = btn.textContent;
    btn.textContent = "Copied!";
    setTimeout(() => btn.textContent = old, 2000);
}
